from detectors.fileTools import _readFile
from detectors.detectorContext import DetectorContext, DetectorException

import os
import fnmatch
import re


class Evidence:
    def __init__(self):
        self.modelElementDirectories = []
    def check(self, project):
        pass

class ProjectDirectoriesPresent(Evidence):
    def __init__(self, directoryList):
        super().__init__()
        if not isinstance(directoryList, list):
            directoryList = [directoryList]
        self.directoryList = directoryList

    def check(self, project, modelElementName):
        failedEvidences = []
        for dir in self.directoryList:
            realPathDir = os.path.realpath(os.path.join(project.rootDirectory, dir))
            print("project dir real path = " + realPathDir)
            if not realPathDir in project.allDirectories:
                failedEvidences.append(f"evidence '{self.__class__.__name__}' failed for directory '{dir}' required for model element '{modelElementName}'")
            else:
                self.modelElementDirectories.append(dir)
        return failedEvidences

class FilesPresent(Evidence):
    def __init__(self, fileList):
        super().__init__()
        if not isinstance(fileList, list):
            fileList = [fileList]
        self.fileList = fileList
        self.filesRealPaths = []

    def check(self, project, modelElementName):
        failedEvidences = []
        for evidenceDir in self.modelElementDirectories:
            evidenceRootDir = os.path.join(project.rootDirectory, evidenceDir)
            for file in self.fileList:
                realPathFile = os.path.realpath(os.path.join(evidenceRootDir, file))
                if not realPathFile in project.allFiles:
                    failedEvidences.append(f"evidence '{self.__class__.__name__}' failed for file '{file}' required for model element '{modelElementName}': file not found")
                self.filesRealPaths.append(realPathFile)
        return failedEvidences


class FileContains(FilesPresent):
    def __init__(self, file, rules):
        super().__init__(file)
        self.rules = rules

    def check(self, project, modelElementName):
        # check that file is present
        failedEvidences = super().check(project, modelElementName)
        if failedEvidences != []:
            return failedEvidences
        
        content = _readFile(self.filesRealPaths[0])

        # check file rules
        for rule in self.rules:
            print("NEED TO CHECK RULE: " + rule + " for file " + self.filesRealPaths[0])
            if re.search(rule, content) == None:
                failedEvidences.append(f"evidence '{self.__class__.__name__}' failed for rule '{rule}' checked in file '{self.fileList[0]}' required for model element '{modelElementName}'")
    
        return failedEvidences

class DetectInFile(FilesPresent):
    def __init__(self, file, detectorsFunction, **kwargs):
        super().__init__(file)
        self.detectorsFunction = detectorsFunction
        self.kwargs = kwargs

    def check(self, project, modelElementName):
        failedEvidences = super().check(project, modelElementName)
        if failedEvidences != []:
            return failedEvidences
        content = _readFile(self.filesRealPaths[0])
        detectorContext = DetectorContext(content)
        try:
            self.detectorsFunction(detectorContext, **self.kwargs)
        except DetectorException as e:
            failedEvidences.append(f"evidence '{self.__class__.__name__}' failed for model element '{modelElementName}': " + str(e))
        return failedEvidences    

class AtLeastOnePresent(Evidence):
    # accepts UNIX-style patterns, as matched by fnmatch
    # checks that for each path pattern at least one file is present in the project that matches the pattern
    def __init__(self, filePathPatterns):
        super().__init__()
        if not isinstance(filePathPatterns, list):
            filePathPatterns = [filePathPatterns]
        self.filePathPatterns = filePathPatterns

    def check(self, project, modelElementName):
        failedEvidences = []
        for evidenceDir in self.modelElementDirectories:
            evidenceRootDir = os.path.join(project.rootDirectory, evidenceDir)
            for pathPattern in self.filePathPatterns:
                realPath = os.path.realpath(os.path.join(evidenceRootDir, pathPattern))
                print("checking: " + realPath)
                match = False
                for projectPath in project.allPaths:
                    if fnmatch.fnmatch(projectPath, realPath):
                        match = True
                        break
                if not match:
                    failedEvidences.append(f"evidence '{self.__class__.__name__}' failed for path pattern '{pathPattern}' required for model element '{modelElementName}'")
        return failedEvidences

class NoNewFailedEvidences(Evidence):
    def __init__(self, numberOfFailedEvidencesBefore, reason):
        super().__init__()
        self.numberOfFailedEvidencesBefore = numberOfFailedEvidencesBefore
        self.reason = reason
    def check(self, project, modelElementName):
        failedEvidences = []
        if len(project.failedEvidences) > self.numberOfFailedEvidencesBefore:
            failedEvidences.append(f"evidence '{self.__class__.__name__}' required for model element '{modelElementName}' failed because some '{self.reason}' failed before")
        return failedEvidences
     




